import pandas as pd
import matplotlib.pyplot as plt

# Load the CSV file
file_path = 'results.csv'  # Update the path to where your file is stored
df = pd.read_csv(file_path)

# Adjusting the column names based on the contents
df.columns = ['Client_Round', 'mu', 'B', 'H']

# Replace NaN values with 0
df.fillna(0, inplace=True)

# Converting 'H' to numeric (already handled since NaNs are replaced with 0)
df['H'] = pd.to_numeric(df['H'], errors='coerce')

# Filtering the data to only include the first 90 rounds
df['Round'] = df['Client_Round'].apply(lambda x: int(x.split('_')[-1]))
filtered_df = df[df['Round'] <= 90]

# Grouping by 'Round' and calculating min, max, and mean for B and H
grouped_df = filtered_df.groupby('Round').agg({
    'B': ['min', 'max', 'mean'],
    'H': ['min', 'max', 'mean']
}).reset_index()

# Rename the columns for easier access
grouped_df.columns = ['Round', 'B_min', 'B_max', 'B_mean', 'H_min', 'H_max', 'H_mean']

# Custom font sizes
label_fontsize = 30  # Axis label font size
tick_fontsize = 25  # Tick font size
legend_fontsize = 20  # Legend font size

# Set font and figure layout
plt.rcParams['font.family'] = 'Times New Roman'  # Set font to Times New Roman
plt.rcParams['figure.autolayout'] = True  # Auto layout to prevent content overflow

# Plotting B and H without the range legend
plt.figure(figsize=(10, 6))

# Shaded area for B (no label for range)
plt.fill_between(grouped_df['Round'], grouped_df['B_min'], grouped_df['B_max'], color='green', alpha=0.2)

# B mean curve
plt.plot(grouped_df['Round'], grouped_df['B_mean'], color='green', label='B')

# Shaded area for H (no label for range)
plt.fill_between(grouped_df['Round'], grouped_df['H_min'], grouped_df['H_max'], color='blue', alpha=0.2)

# H mean curve
plt.plot(grouped_df['Round'], grouped_df['H_mean'], color='blue', label='H')

# Customize labels and fonts
plt.xlabel('Interations T', fontsize=label_fontsize)
plt.ylabel('B & H Value', fontsize=label_fontsize)
plt.xticks(fontsize=tick_fontsize)
plt.yticks(fontsize=tick_fontsize)
plt.legend(fontsize=legend_fontsize)
plt.grid(True)

# Save the figure
plt.savefig('B_H_plot.pdf', format='pdf')  # Save as PDF
plt.savefig('B_H_plot.png', format='png', dpi=300)  # Save as PNG with 300 dpi

plt.show()

# Grouping by 'Round' and calculating min, max, and mean for μ
mu_grouped_df = filtered_df.groupby('Round').agg({
    'mu': ['min', 'max', 'mean']
}).reset_index()

# Rename the columns for easier access
mu_grouped_df.columns = ['Round', 'mu_min', 'mu_max', 'mu_mean']

# Plotting μ without the range legend
plt.figure(figsize=(10, 6))

# Shaded area for μ (no label for range)
plt.fill_between(mu_grouped_df['Round'], mu_grouped_df['mu_min'], mu_grouped_df['mu_max'], color='red', alpha=0.2)

# μ mean curve
plt.plot(mu_grouped_df['Round'], mu_grouped_df['mu_mean'], color='red', label='μ')

# Customize labels and fonts
plt.xlabel('Interations T', fontsize=label_fontsize)
plt.ylabel('μ Value', fontsize=label_fontsize)
plt.xticks(fontsize=tick_fontsize)
plt.yticks(fontsize=tick_fontsize)
plt.legend(fontsize=legend_fontsize)
plt.grid(True)

# Save the figure
plt.savefig('mu_plot.pdf', format='pdf')  # Save as PDF
plt.savefig('mu_plot.png', format='png', dpi=300)  # Save as PNG with 300 dpi

plt.show()
